/*
 * Query the FMS for all outstanding alerts
 */
#include <time.h>
#include <stdio.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_alert.h"
#include "lf_fms.h"

void dump_alerts(int fms, int show_acked, int show_relics);

static void
usage()
{
  fprintf(stderr, "Print a list of active alerts.  By default, this prints only\n");
  fprintf(stderr, "alerts which have not been ACKed and are not relics. (See appendix\n");
  fprintf(stderr, "on \"Alerts\" in the README).\n");
  fprintf(stderr, "Each alert has a unique index which can be passed to fm_ack_alert\n");
  fprintf(stderr, "to acknowledge the alert.\n\n");
  fprintf(stderr, "Usage: fm_show_alerts\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -a - show ACKed alerts also (marked with [A])\n");
  fprintf(stderr, "  -r - show relic alerts also (marked with [R])\n");
  fprintf(stderr, "  -s <fms_server> - specify fms server\n");
  fprintf(stderr, "  -V - print version\n");
  exit(1);
}

int
main(
  int argc,
  char **argv)
{
  int c;
  char *fms_names;
  int fms;
  int show_acked;
  int show_relics;
  extern char *optarg;

  /* Initialize libfma */
  lf_init();

  /* defaults */
  fms_names = NULL;
  show_acked = FALSE;
  show_relics = FALSE;

  /*
   * Get command line args
   */
  while ((c = getopt(argc, argv, "hs:arV")) != EOF) switch (c) {
  case 'h':
    usage();
    break;
  case 's':
    fms_names = optarg;
    break;
  case 'a':
    show_acked = TRUE;
    break;
  case 'r':
    show_relics = TRUE;
    break;
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  }

  /*
   * Open connection to fms
   */
  fms = lf_fms_connect(fms_names);
  if (fms == -1) {
    fprintf(stderr, "Error connecting to ");
    if (fms_names == NULL) {
      fprintf(stderr, "default FMS\n");
    } else {
      fprintf(stderr, "FMS in (%s)\n", fms_names);
    }
    exit(1);
  }

  /*
   * Dump the alerts
   */
  dump_alerts(fms, show_acked, show_relics);

  /* all done, shut down connection */
  lf_fms_disconnect(fms);

  exit(0);
}

/*
 * Query the FMS for all outstanding alerts
 */
void
dump_alerts(
  int fms,
  int show_acked,
  int show_relics)
{
  int num_alerts;
  struct lf_alert *alerts;
  struct lf_alert *ap;
  lf_string_t buf;
  int i;

  /* get the list of alerts from the FMS */
  num_alerts = lf_fms_query_alerts(fms, show_acked, show_relics, &alerts);

  /* dump the alerts */
  for (i=0; i<num_alerts; ++i) {
    ap = alerts + i;

    /* alert ID */
    printf("%6d ", ap->alert_id);

    /* print alert time */
    {
      char *s;
      time_t t;
      t = ap->alert_time;
      s = asctime(localtime(&t));
      s[strlen(s)-1] = '\0';
      printf("%s ", s);
    }
    
    if (show_acked) {
      printf("%s ", ap->acked ? "[A]" : "   ");
    }
    if (show_relics) {
      printf("%s ", ap->relic ? "[R]" : "   ");
    }

    lf_alert_string(buf, ap);
    printf("%s\n", buf);
  }
}
